@tool
extends EditorPlugin

var _btn: Button = null

func _enter_tree() -> void:
	if _btn == null:
		_btn = Button.new()
		_btn.text = "Fit to Game Camera"
		_btn.tooltip_text = "Make the selected plane fill the CURRENT Camera3D view at game resolution."
		_btn.pressed.connect(_on_fit_pressed)
		add_control_to_container(CONTAINER_SPATIAL_EDITOR_MENU, _btn)
		print("[FitPlaneToGameCamera] Loaded.")

func _exit_tree() -> void:
	if _btn != null:
		remove_control_from_container(CONTAINER_SPATIAL_EDITOR_MENU, _btn)
		_btn.queue_free()
		_btn = null
		print("[FitPlaneToGameCamera] Unloaded.")

func _on_fit_pressed() -> void:
	var mesh_inst: MeshInstance3D = _get_selected_mesh_instance()
	if mesh_inst == null:
		_push_warn("Select a MeshInstance3D (your plane) first.")
		return

	var cam: Camera3D = _find_current_camera()
	if cam == null:
		_push_warn("No Camera3D with 'current' enabled found in the scene.")
		return

	var game_size: Vector2i = _get_game_resolution()
	if game_size.x <= 0 or game_size.y <= 0:
		_push_warn("Invalid project game resolution.")
		return

	var target_aspect: float = float(game_size.x) / float(game_size.y)
	var proj: int = cam.projection

	# Make plane face the camera (billboard-style), then fit.
	_face_plane_to_camera(mesh_inst, cam)

	# Ensure plane aspect matches the game aspect (adjust X scale only).
	_match_plane_aspect(mesh_inst, target_aspect)

	# Now place/scale depending on camera projection.
	if proj == Camera3D.PROJECTION_PERSPECTIVE:
		_fit_perspective(mesh_inst, cam)
	elif proj == Camera3D.PROJECTION_ORTHOGONAL:
		_fit_orthogonal(mesh_inst, cam, target_aspect)
	else:
		_push_warn("Unsupported camera projection (frustum mode). Use Perspective or Orthogonal.")
		return

	# Small forward nudge to avoid z-fighting with near clip edge.
	_nudge_forward(mesh_inst, cam, 0.001)

	# Try to make front-face visible regardless of winding.
	_disable_backface_cull_if_standard_material(mesh_inst)

	_push_ok("Fitted plane to current game camera.")

func _get_selected_mesh_instance() -> MeshInstance3D:
	var sel: EditorSelection = get_editor_interface().get_selection()
	var nodes: Array = sel.get_selected_nodes()
	for n in nodes:
		if n is MeshInstance3D:
			return n
	return null

func _find_current_camera() -> Camera3D:
	var root: Node = get_editor_interface().get_edited_scene_root()
	if root == null:
		return null
	var stack: Array[Node] = [root]
	while stack.size() > 0:
		var n: Node = stack.pop_back()
		if n is Camera3D:
			var c: Camera3D = n as Camera3D
			if c.current == true:
				return c
		for child in n.get_children():
			stack.push_back(child)
	return null

func _get_game_resolution() -> Vector2i:
	var w: int = int(ProjectSettings.get_setting("display/window/size/viewport_width"))
	var h: int = int(ProjectSettings.get_setting("display/window/size/viewport_height"))
	return Vector2i(w, h)

func _face_plane_to_camera(plane: MeshInstance3D, cam: Camera3D) -> void:
	# Align plane so it visually faces the camera and stays upright with the camera.
	var cam_xform: Transform3D = cam.global_transform
	plane.global_transform = cam_xform
	# Flip 180° around camera up so the plane front faces the camera (common Godot mesh forward is -Z).
	plane.rotate_object_local(Vector3.UP, PI)

func _get_plane_base_size_xy(plane: MeshInstance3D) -> Vector2:
	# Returns the unscaled width/height in world units as defined by mesh resource.
	var m: Mesh = plane.mesh
	if m is QuadMesh:
		var qm: QuadMesh = m as QuadMesh
		return Vector2(qm.size.x, qm.size.y)
	# Fallback: use AABB in local space (best effort).
	var aabb: AABB = m.get_aabb()
	return Vector2(aabb.size.x, aabb.size.y)

func _get_plane_world_size_xy(plane: MeshInstance3D) -> Vector2:
	var base_xy: Vector2 = _get_plane_base_size_xy(plane)
	var sc: Vector3 = plane.scale
	return Vector2(abs(base_xy.x * sc.x), abs(base_xy.y * sc.y))

func _match_plane_aspect(plane: MeshInstance3D, target_aspect: float) -> void:
	var size_xy: Vector2 = _get_plane_world_size_xy(plane)
	if size_xy.y <= 0.0:
		return
	var current_aspect: float = size_xy.x / size_xy.y
	var epsilon: float = 0.0001
	if abs(current_aspect - target_aspect) < epsilon:
		return
	# Adjust X scale so width = height * target_aspect
	var sc: Vector3 = plane.scale
	var base_xy: Vector2 = _get_plane_base_size_xy(plane)
	if base_xy.x == 0.0 or base_xy.y == 0.0:
		return
	var height_world: float = abs(base_xy.y * sc.y)
	var desired_width: float = height_world * target_aspect
	var new_scale_x: float = desired_width / abs(base_xy.x)
	if sc.x < 0.0:
		new_scale_x = -new_scale_x
	sc.x = new_scale_x
	plane.scale = sc

func _fit_perspective(plane: MeshInstance3D, cam: Camera3D) -> void:
	# For perspective, distance so that plane height matches frustum height at that distance:
	# height_at_d = 2 * d * tan(fov_v/2)
	# d = plane_height / (2 * tan(fov_v/2))
	var fov_v_deg: float = cam.fov
	var fov_v_rad: float = deg_to_rad(fov_v_deg)
	var size_xy: Vector2 = _get_plane_world_size_xy(plane)
	var plane_height: float = size_xy.y
	if plane_height <= 0.0:
		_push_warn("Plane height is zero; cannot fit.")
		return
	var denom: float = 2.0 * tan(fov_v_rad * 0.5)
	if denom == 0.0:
		_push_warn("Camera FOV invalid; cannot fit.")
		return
	var distance: float = plane_height / denom
	# Position the plane directly in front of the camera at this distance.
	var forward: Vector3 = -cam.global_transform.basis.z.normalized()
	var target_pos: Vector3 = cam.global_transform.origin + forward * distance
	var xf: Transform3D = plane.global_transform
	xf.origin = target_pos
	plane.global_transform = xf

func _fit_orthogonal(plane: MeshInstance3D, cam: Camera3D, target_aspect: float) -> void:
	# For orthographic cameras, the camera 'size' is the vertical span in world units.
	# Make plane height = size, width = size * aspect. Scale X accordingly, keep Y scale for height.
	var size_xy_before: Vector2 = _get_plane_world_size_xy(plane)
	var base_xy: Vector2 = _get_plane_base_size_xy(plane)
	if base_xy.x == 0.0 or base_xy.y == 0.0:
		_push_warn("Plane base size invalid; cannot fit.")
		return
	var sc: Vector3 = plane.scale
	var desired_height: float = cam.size
	if desired_height <= 0.0:
		_push_warn("Camera size invalid; cannot fit.")
		return
	# Set Y scale to get the desired height exactly.
	var new_scale_y: float = desired_height / abs(base_xy.y)
	if sc.y < 0.0:
		new_scale_y = -new_scale_y
	sc.y = new_scale_y
	# Compute desired width from aspect and adjust X scale.
	var desired_width: float = desired_height * target_aspect
	var new_scale_x: float = desired_width / abs(base_xy.x)
	if sc.x < 0.0:
		new_scale_x = -new_scale_x
	sc.x = new_scale_x
	plane.scale = sc

	# Place plane somewhere in front of camera, within near/far. Distance does not affect framing in ortho.
	var forward: Vector3 = -cam.global_transform.basis.z.normalized()
	var dist: float = max(cam.near + 0.1, 0.1)
	var pos: Vector3 = cam.global_transform.origin + forward * dist
	var xf: Transform3D = plane.global_transform
	xf.origin = pos
	plane.global_transform = xf

func _nudge_forward(plane: MeshInstance3D, cam: Camera3D, amount: float) -> void:
	if amount <= 0.0:
		return
	var forward: Vector3 = -cam.global_transform.basis.z.normalized()
	var xf: Transform3D = plane.global_transform
	xf.origin = xf.origin + forward * amount
	plane.global_transform = xf

func _disable_backface_cull_if_standard_material(plane: MeshInstance3D) -> void:
	# Try to make sure it shows regardless of winding; only touches StandardMaterial3D on surface 0.
	var mat: Material = plane.get_active_material(0)
	if mat is StandardMaterial3D:
		var sm: StandardMaterial3D = mat as StandardMaterial3D
		sm.cull_mode = BaseMaterial3D.CULL_DISABLED

func _push_warn(msg: String) -> void:
	push_warning("[FitPlaneToGameCamera] " + msg)
	print("[FitPlaneToGameCamera][WARN] " + msg)

func _push_ok(msg: String) -> void:
	print("[FitPlaneToGameCamera] " + msg)
